// ZEN Study Plus - Dynamic Island for N高等学校サイト
// MacノッチスタイルのダイナミックアイランドUI

(function() {
  'use strict';

class DynamicIsland {
  constructor() {
    this.islandElement = null;
    this.isActive = false;
    this.lockState = false;
    this.currentPanel = 'main';
    this.isLoggedIn = false;
    
    // タイマー設定
    this.timerMode = 'manual'; // manual, pomodoro, 45-15
    this.timerState = { active: false }; // backgroundからのstatusを保持
    
    // 音楽設定
    this.musicState = {
      isPlaying: false,
      currentTrackIndex: 0,
      volume: 50,
      tracks: [
        { title: 'First Snow', file: '01_First_Snow.mp3' },
        { title: 'Laundry On The Wire', file: '02_Laundry_On_The_Wire.mp3' },
        { title: 'Everything You Ever Dreamed', file: '03_Everything_You_Ever_Dreamed.mp3' },
        { title: 'Keeping Cool', file: '04_Keeping_Cool.mp3' },
        { title: 'Snow Drift', file: '05_Snow_Drift.mp3' },
        { title: 'Windows Down', file: '06_Windows_Down.mp3' },
        { title: 'Hour Delay', file: '07_Hour_Delay.mp3' }
      ]
    };
    
    // Spotify 簡易状態（コンパクト表示用）
    this.spotifyCompact = {
      isPlaying: false,
      title: '',
      artist: '',
      cover: ''
    };
    
    // 表示用ローカルチック（メッセージ未到達時も滑らかに更新）
    this._displayAnchor = {
      ts: 0,                // 最終同期のエポックms
      manualElapsedMs: 0,   // 手動モードの同期時点の経過ms
      advRemainSec: 0       // 高度モードの同期時点の残り秒
    };
    
    // mouseleave時の遅延用タイマー
    this.mouseleaveTimer = null;
    
    // ホバー関連のタイマーと状態
    this.hoverTimer = null;
    this.hoverActive = false;
    
    this.init();
  }

  init() {
    this.createStyles();
    this.createIsland();
    this.setupEventListeners();
    this.loadSettings();
    this.listenForBroadcasts();
    // 常時ポーリングしてコンパクト表示も更新（軽量な1秒周期）
    this.startPolling();
    // ローカル表示用チック（1秒）
    this.startDisplayTicker();
  }

  createStyles() {
    const musicIconUrl = chrome.runtime.getURL('images/music.png');
    const style = document.createElement('style');
    style.textContent = `
      :root {
        --island-notch-w: 236px; /* slightly wider than original, but narrower than 260 */
        --island-notch-h: 38px;
        --island-w: min(760px, 90vw);
        --island-h: clamp(200px, 42vh, 460px);
        --island-bg: rgba(10,10,12,.98);
        --island-border: rgba(255,255,255,.08);
        --island-text: #e9eaec;
        --island-accent: #007AFF;
        --island-secondary: rgba(255,255,255,.06);
        --island-success: #32D74B;
        --island-warning: #FF9F0A;
        --island-danger: #FF453A;
      }

      #zsp-dynamic-island {
        position: fixed;
        top: 0;
        left: 50%;
        transform: translateX(-50%);
        width: var(--island-notch-w);
        height: var(--island-notch-h);
        border-radius: 0 0 14px 14px;
        background: #000;
        border: 1px solid var(--island-border);
        border-top: none;
        box-shadow: 0 10px 30px rgba(0,0,0,.55), inset 0 0 0 1px rgba(255,255,255,.04);
        overflow: hidden;
        z-index: 999999;
        display: grid;
        place-items: center;
        font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", "SF Pro Display", Roboto, "Helvetica Neue", Arial, sans-serif;
        color: var(--island-text);
        font-size: 14px;
        font-weight: 400;
        letter-spacing: -0.01em;
        -webkit-font-smoothing: antialiased;
        -moz-osx-font-smoothing: grayscale;
        transition: width 1.2s cubic-bezier(.2,.8,.2,1),
                    height 1.2s cubic-bezier(.2,.8,.2,1),
                    border-radius 1.2s cubic-bezier(.2,.8,.2,1),
                    background .6s cubic-bezier(.2,.8,.2,1),
                    box-shadow 1.2s cubic-bezier(.2,.8,.2,1);
      }

      #zsp-dynamic-island[data-active="true"] {
        width: var(--island-w);
        height: var(--island-h);
        background: var(--island-bg);
        border: 1px solid var(--island-border);
        border-top: none;
        border-radius: 0 0 22px 22px;
        backdrop-filter: saturate(160%) blur(20px);
      }

      .zsp-island-compact {
        font-size: 12px;
        opacity: .85;
        white-space: nowrap;
        -webkit-user-select: none;
        -moz-user-select: none;
        -ms-user-select: none;
        user-select: none;
      }

      .zsp-island-panel {
        width: 100%;
        height: 100%;
        opacity: 0;
        pointer-events: none;
        display: flex;
        flex-direction: column;
        gap: 12px;
        padding: 14px;
        transition: opacity .8s cubic-bezier(.2,.8,.2,1) .4s;
        overflow-y: auto;
        /* スクロールバーを非表示 */
        scrollbar-width: none; /* Firefox */
        -ms-overflow-style: none; /* Internet Explorer 10+ */
      }

      .zsp-island-panel::-webkit-scrollbar {
        display: none; /* WebKit */
      }

      #zsp-dynamic-island[data-active="true"] .zsp-island-panel {
        opacity: 1;
        pointer-events: auto;
        transition: opacity .6s cubic-bezier(.2,.8,.2,1) .3s;
      }

      .zsp-panel-nav {
        display: flex;
        gap: 8px;
        background: var(--island-secondary);
        border-radius: 12px;
        padding: 4px;
      }

      .zsp-nav-btn {
        padding: 8px 16px;
        border: none;
        background: transparent;
        color: var(--island-text);
        border-radius: 8px;
        font-size: 12px;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.2s ease;
        opacity: 0.7;
        letter-spacing: -0.01em;
      }

      .zsp-nav-btn.active {
        background: var(--island-accent);
        opacity: 1;
      }

      .zsp-nav-btn:hover {
        opacity: 1;
      }

      .zsp-panel-content {
        flex: 1;
        display: none;
      }

      .zsp-panel-content.active {
        display: block;
      }

      .zsp-control-group {
        background: var(--island-secondary);
        border: 1px solid var(--island-border);
        padding: 12px;
        border-radius: 14px;
        margin-bottom: 10px;
        position: relative; /* for overlay positioning */
      }

      .zsp-control-group h3 {
        margin: 0 0 8px 0;
        font-size: 15px;
        font-weight: 600;
        letter-spacing: -0.02em;
      }

      /* Login guard overlay */
      .zsp-login-overlay {
        position: absolute;
        inset: 8px;
        display: none;
        align-items: center;
        justify-content: center;
        text-align: center;
        background: rgba(0,0,0,0.55);
        border: 1px solid var(--island-border);
        border-radius: 12px;
        z-index: 2;
        padding: 16px;
        backdrop-filter: blur(6px);
      }

      .zsp-login-overlay .zsp-overlay-content h3 {
        margin: 0 0 6px 0;
        font-size: 14px;
        font-weight: 600;
      }

      .zsp-login-overlay .zsp-overlay-content p {
        margin: 0 0 10px 0;
        font-size: 12px;
        opacity: .85;
      }

      .zsp-login-blocked {
        opacity: .6;
      }
      .zsp-login-blocked .zsp-btn,
      .zsp-login-blocked .zsp-control-btn,
      .zsp-login-blocked .zsp-timer-mode-btn {
        pointer-events: none;
        filter: grayscale(40%);
      }

      /* Ensure overlay actions remain clickable */
      .zsp-login-blocked .zsp-login-overlay,
      .zsp-login-blocked .zsp-login-overlay * {
        pointer-events: auto !important;
        filter: none;
      }

      .zsp-control-row {
        display: flex;
        align-items: center;
        gap: 12px;
        margin-bottom: 8px;
        flex-wrap: wrap;
      }

      .zsp-control-row:last-child {
        margin-bottom: 0;
      }

      .zsp-control-row .zsp-track-title {
        flex: 1;
        font-size: 12px;
        font-weight: 500;
        min-width: 0;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
      }

      .zsp-btn {
        border: 1px solid var(--island-border);
        border-bottom-width: 2px;
        padding: 8px 16px;
        border-radius: 10px;
        font-size: 13px;
        background: var(--island-secondary);
        color: var(--island-text);
        cursor: pointer;
        transition: all 0.2s ease;
        font-weight: 500;
        letter-spacing: -0.01em;
        font-family: inherit;
      }

      .zsp-btn:hover {
        background: rgba(255,255,255,.12);
      }

      .zsp-btn.primary {
        background: var(--island-accent);
        border-color: var(--island-accent);
      }

      .zsp-btn.danger {
        background: var(--island-danger);
        border-color: var(--island-danger);
        color: white;
      }

      .zsp-btn.success {
        background: var(--island-success);
        border-color: var(--island-success);
        color: white;
      }

      .zsp-slider-container {
        flex: 1;
        display: flex;
        align-items: center;
        gap: 8px;
      }

      .zsp-slider {
        -webkit-appearance: none;
        appearance: none;
        flex: 1;
        height: 4px;
        background: var(--island-secondary);
        border-radius: 2px;
        outline: none;
      }

      .zsp-slider::-webkit-slider-thumb {
        -webkit-appearance: none;
        appearance: none;
        width: 18px;
        height: 18px;
        background: var(--island-accent);
        border-radius: 50%;
        cursor: pointer;
      }

      .zsp-slider::-moz-range-thumb {
        width: 18px;
        height: 18px;
        background: var(--island-accent);
        border-radius: 50%;
        cursor: pointer;
        border: none;
      }

      .zsp-select {
        background: var(--island-secondary);
        border: 1px solid var(--island-border);
        color: var(--island-text);
        padding: 8px 12px;
        border-radius: 8px;
        font-size: 13px;
        min-width: 120px;
      }

      .zsp-timer-display {
        text-align: center;
        padding: 16px;
        background: var(--island-secondary);
        border-radius: 14px;
        margin-bottom: 12px;
      }

      .zsp-timer-time {
        font-size: 28px;
        font-weight: 300;
        letter-spacing: 1px;
        color: var(--island-accent);
        margin-bottom: 6px;
      }

      .zsp-timer-label {
        font-size: 12px;
        opacity: 0.7;
      }

      #zsp-island-sensor {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        height: 110px;
        pointer-events: none;
        z-index: 999998;
      }

      .zsp-status-indicator {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        font-size: 11px;
        opacity: 0.8;
        -webkit-user-select: none;
        -moz-user-select: none;
        -ms-user-select: none;
        user-select: none;
      }

      .zsp-compact-cover {
        width: 16px;
        height: 16px;
        border-radius: 3px;
        object-fit: cover;
        background: #1c1c1e;
        border: 1px solid var(--island-border);
        transition: opacity 0.2s ease;
      }
      
      .zsp-compact-cover:not([src]),
      .zsp-compact-cover[src=""] {
        background: url('${musicIconUrl}') center / cover;
        background-size: contain;
      }

      /* Compact music title marquee */
      .zsp-compact-title {
        display: inline-block;
        max-width: 176px; /* fits within notch width with dot and icon */
        overflow: hidden;
        vertical-align: middle;
        -webkit-user-select: none;
        -moz-user-select: none;
        -ms-user-select: none;
        user-select: none;
      }
      .zsp-marquee {
        display: inline-block;
      }
      .zsp-marquee-track {
        display: inline-flex;
        align-items: center;
        gap: 32px;
        will-change: transform;
      }
      .zsp-compact-title.too-long .zsp-marquee-track {
        animation: zsp-marquee 12s linear infinite;
      }
      .zsp-marquee-text {
        white-space: nowrap;
      }
      @keyframes zsp-marquee {
        0% { transform: translateX(0); }
        100% { transform: translateX(-50%); }
      }

      .zsp-status-dot {
        width: 6px;
        height: 6px;
        border-radius: 50%;
        background: #32D74B;
      }

      .zsp-status-dot.inactive {
        background: #8E8E93;
      }

      /* タイマーモードボタン */
      .zsp-timer-mode-buttons {
        display: flex;
        gap: 4px;
        background: var(--island-secondary);
        border-radius: 8px;
        padding: 4px;
        margin-bottom: 8px;
      }

      .zsp-timer-mode-btn {
        flex: 1;
        padding: 6px 12px;
        border: none;
        background: transparent;
        color: var(--island-text);
        border-radius: 6px;
        font-size: 11px;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.2s ease;
        opacity: 0.7;
      }

      .zsp-timer-mode-btn.active {
        background: var(--island-accent);
        opacity: 1;
      }

      .zsp-music-mode-btn {
        flex: 1;
        padding: 6px 12px;
        border: none;
        background: transparent;
        color: var(--island-text);
        border-radius: 6px;
        font-size: 11px;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.2s ease;
        opacity: 0.7;
      }

      .zsp-music-mode-btn.active {
        background: var(--island-accent);
        opacity: 1;
      }

      .zsp-music-mode-btn:hover {
        opacity: 1;
      }

      /* 音楽プレイヤー */
      .zsp-music-player {
        margin-bottom: 12px;
      }

      .zsp-track-info {
        text-align: center;
        margin-bottom: 12px;
        padding: 12px 0;
        border-bottom: 1px solid var(--island-border);
      }

      .zsp-track-title {
        font-weight: 500;
        font-size: 13px;
        margin-bottom: 4px;
        color: var(--island-text);
      }

      .zsp-track-number {
        font-size: 11px;
        opacity: 0.7;
      }

      /* Spotify thumbnail */
      .zsp-spotify-cover {
        width: 40px;
        height: 40px;
        border-radius: 8px;
        object-fit: cover;
        background: #1c1c1e;
        border: 1px solid var(--island-border);
        display: block;
      }
      
      .zsp-spotify-cover.hidden {
        display: none;
      }

      .zsp-player-controls {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
        flex-wrap: wrap;
      }

      .zsp-player-controls .zsp-btn {
        min-width: 36px;
        padding: 6px 12px;
        font-size: 12px;
      }

      .zsp-player-controls .zsp-btn.primary {
        min-width: 42px;
        padding: 8px 14px;
        font-size: 14px;
      }

      .zsp-control-btn {
        min-width: 40px;
        height: 40px;
        border: 1px solid var(--island-border);
        border-radius: 50%;
        background: var(--island-secondary);
        color: var(--island-text);
        cursor: pointer;
        transition: all 0.2s ease;
        font-size: 14px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-bottom-width: 2px;
      }

      .zsp-control-btn:hover {
        background: rgba(255,255,255,.12);
        transform: translateY(-1px);
      }

      .zsp-control-btn:active {
        transform: translateY(0);
      }

      .zsp-control-btn.primary {
        background: var(--island-accent);
        border-color: var(--island-accent);
        color: white;
        font-size: 16px;
        min-width: 46px;
        height: 46px;
      }

      .zsp-volume-control {
        position: relative;
        display: flex;
        align-items: center;
      }

      .zsp-volume-slider-container {
        position: absolute;
        bottom: 100%;
        left: 50%;
        transform: translateX(-50%);
        background: var(--island-bg);
        border: 1px solid var(--island-border);
        border-radius: 8px;
        padding: 12px 8px;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 8px;
        opacity: 0;
        pointer-events: none;
        transition: opacity 0.3s ease;
        margin-bottom: 8px;
        backdrop-filter: blur(20px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        z-index: 1000;
      }

      .zsp-volume-control:hover .zsp-volume-slider-container,
      .zsp-volume-slider-container:hover {
        opacity: 1;
        pointer-events: auto;
      }

      .zsp-volume-slider {
        -webkit-appearance: none;
        appearance: none;
        width: 4px;
        height: 80px;
        background: var(--island-secondary);
        border-radius: 2px;
        outline: none;
        writing-mode: vertical-lr;
        direction: rtl;
      }

      .zsp-volume-slider::-webkit-slider-thumb {
        -webkit-appearance: none;
        appearance: none;
        width: 16px;
        height: 16px;
        background: var(--island-accent);
        border-radius: 50%;
        cursor: pointer;
      }

      .zsp-volume-slider::-moz-range-thumb {
        width: 16px;
        height: 16px;
        background: var(--island-accent);
        border-radius: 50%;
        cursor: pointer;
        border: none;
      }

      .zsp-volume-display {
        font-size: 11px;
        min-width: 30px;
        text-align: center;
        opacity: 0.8;
      }

      .zsp-volume-icon-svg {
        width: 20px;
        height: 20px;
        fill: currentColor;
      }

      /* 環境音アイテム */
      .zsp-ambient-item {
        margin-bottom: 12px;
      }

      .zsp-ambient-info {
        display: flex;
        align-items: center;
        gap: 12px;
      }

      .zsp-ambient-info > span {
        min-width: 120px;
        font-size: 12px;
      }

      .zsp-volume-value {
        min-width: 36px;
        font-size: 11px;
        text-align: right;
      }

      /* Windows環境での追加調整 */
      @supports (-ms-ime-align: auto) {
        /* Edge/IE向け */
        #zsp-dynamic-island {
          font-weight: 400;
        }
        .zsp-control-group h3 {
          font-weight: 600;
        }
        .zsp-btn, .zsp-nav-btn {
          font-weight: 500;
        }
      }

      /* Windows Chrome特有の調整 */
      @media (-webkit-min-device-pixel-ratio: 0) {
        .zsp-timer-time {
          font-weight: 200;
        }
        .zsp-control-group h3 {
          font-weight: 600;
        }
      }

      @media (prefers-reduced-motion: reduce) {
        #zsp-dynamic-island,
        .zsp-island-panel {
          transition: none;
        }
      }

      @media (max-width: 768px) {
        :root {
          --island-w: 95vw;
          --island-h: 60vh;
        }
      }
    `;
    document.head.appendChild(style);
  }

  createIsland() {
    // センサー要素
    const sensor = document.createElement('div');
    sensor.id = 'zsp-island-sensor';
    document.body.appendChild(sensor);

    // アイランド要素
    const island = document.createElement('div');
    island.id = 'zsp-dynamic-island';
    island.setAttribute('aria-label', 'ZEN Study Plus Dynamic Island');
    
    island.innerHTML = `
      <div class="zsp-island-compact">
        <div class="zsp-status-indicator" aria-label="ZEN Study Plus status">
          <div class="zsp-status-dot inactive"></div>
        </div>
      </div>
      
      <div class="zsp-island-panel" aria-hidden="true">
        <div class="zsp-panel-nav">
          <button class="zsp-nav-btn active" data-panel="main">メイン</button>
          <button class="zsp-nav-btn" data-panel="ambient">環境音</button>
        </div>
        
        ${this.createPanelContent()}
      </div>
    `;
    
    document.body.appendChild(island);
    this.islandElement = island;
  }

  createPanelContent() {
    return `
      <!-- メインパネル -->
      <div class="zsp-panel-content active" data-panel="main">
        <div class="zsp-control-group">
          <h3>タイマー</h3>
          <div class="zsp-timer-mode-buttons">
            <button class="zsp-timer-mode-btn active" data-mode="manual">手動</button>
            <button class="zsp-timer-mode-btn" data-mode="pomodoro">ポモドーロ</button>
            <button class="zsp-timer-mode-btn" data-mode="45-15">45・15</button>
          </div>
          <!-- Login required overlay -->
          <div class="zsp-login-overlay" id="zsp-login-overlay" style="display:none;">
            <div class="zsp-overlay-content">
              <h3>ログインが必要です</h3>
              <p>拡張機能アイコンを開き、アカウントでログインしてください。</p>
              <button class="zsp-btn" id="zsp-go-login">ログインページを開く</button>
            </div>
          </div>
          <div class="zsp-timer-display">
            <div class="zsp-timer-time" id="zsp-timer-display">00:00</div>
            <div class="zsp-timer-label" id="zsp-timer-label">集中時間</div>
          </div>
          <div class="zsp-control-row">
            <button class="zsp-btn primary" id="zsp-timer-start">開始</button>
            <button class="zsp-btn danger" id="zsp-timer-stop" style="display: none;">停止</button>
            <button class="zsp-btn" id="zsp-timer-skip" style="display: none;">休憩へスキップ</button>
          </div>
        </div>
        
        <div class="zsp-control-group">
          <h3>ミュージック</h3>
          <div class="zsp-timer-mode-buttons" style="margin-bottom:8px;">
            <button class="zsp-music-mode-btn active" data-music="lofi">Lo‑Fi</button>
            <button class="zsp-music-mode-btn" data-music="spotify">Spotify</button>
          </div>
          <div class="zsp-music-pane" data-pane="lofi" style="display:block;">
          <div class="zsp-control-row">
            <span class="zsp-track-title" id="zsp-track-title">楽曲を選択...</span>
            <div class="zsp-player-controls">
              <button class="zsp-btn" id="zsp-music-prev" title="前の曲">⏮</button>
              <button class="zsp-btn primary" id="zsp-music-play" title="再生/一時停止">▶</button>
              <button class="zsp-btn" id="zsp-music-next" title="次の曲">⏭</button>
              <div class="zsp-volume-control">
                <button class="zsp-control-btn" id="zsp-volume-icon" title="音量">
                  <svg class="zsp-volume-icon-svg" id="zsp-volume-svg" viewBox="0 0 24 24">
                    <path d="M3 9v6h4l5 5V4L7 9H3zm13.5 3c0-1.77-1.02-3.29-2.5-4.03v8.05c1.48-.73 2.5-2.25 2.5-4.02zM14 3.23v2.06c2.89.86 5 3.54 5 6.71s-2.11 5.85-5 6.71v2.06c4.01-.91 7-4.49 7-8.77s-2.99-7.86-7-8.77z"/>
                  </svg>
                </button>
                <div class="zsp-volume-slider-container">
                  <span class="zsp-volume-display" id="zsp-music-volume-value">50%</span>
                  <input type="range" class="zsp-volume-slider" id="zsp-music-volume" min="0" max="100" value="50" orient="vertical">
                </div>
              </div>
            </div>
          </div>
          </div>
          <div class="zsp-music-pane" data-pane="spotify" style="display:none;">
            <div class="zsp-control-row">
              <img class="zsp-spotify-cover" id="zsp-spotify-cover" alt="cover" />
              <span class="zsp-track-title" id="zsp-spotify-title">Spotify設定を確認中...</span>
              <div class="zsp-player-controls">
                <button class="zsp-btn" id="zsp-spotify-connect" title="Spotifyに接続">接続</button>
                <button class="zsp-btn" id="zsp-spotify-prev" title="前へ">⏮</button>
                <button class="zsp-btn primary" id="zsp-spotify-play" title="再生/一時停止">▶</button>
                <button class="zsp-btn" id="zsp-spotify-next" title="次へ">⏭</button>
                <div class="zsp-volume-control">
                  <button class="zsp-control-btn" id="zsp-spotify-volume-icon" title="音量">
                    <svg class="zsp-volume-icon-svg" viewBox="0 0 24 24">
                      <path d="M3 9v6h4l5 5V4L7 9H3zm13.5 3c0-1.77-1.02-3.29-2.5-4.03v8.05c1.48-.73 2.5-2.25 2.5-4.02zM14 3.23v2.06c2.89.86 5 3.54 5 6.71s-2.11 5.85-5 6.71v2.06c4.01-.91 7-4.49 7-8.77s-2.99-7.86-7-8.77z"/>
                    </svg>
                  </button>
                  <div class="zsp-volume-slider-container">
                    <span class="zsp-volume-display" id="zsp-spotify-volume-value">50%</span>
                    <input type="range" class="zsp-volume-slider" id="zsp-spotify-volume" min="0" max="100" value="50" orient="vertical">
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
        
      </div>
      
      <!-- 環境音パネル -->
      <div class="zsp-panel-content" data-panel="ambient">
        <div class="zsp-control-group">
          <div class="zsp-control-row">
            <span>全停止</span>
            <button class="zsp-btn danger" id="zsp-ambient-stop-all">停止</button>
          </div>
        </div>
        
        <div class="zsp-control-group">
          <h3>自然音</h3>
          <div class="zsp-ambient-item">
            <div class="zsp-ambient-info">
              <span>やさしい雨</span>
              <div class="zsp-slider-container">
                <input type="range" class="zsp-slider" data-sound="01_やさしい雨.mp3" min="0" max="100" value="0">
                <span class="zsp-volume-value">0%</span>
              </div>
            </div>
          </div>
          <div class="zsp-ambient-item">
            <div class="zsp-ambient-info">
              <span>心地よい暖炉</span>
              <div class="zsp-slider-container">
                <input type="range" class="zsp-slider" data-sound="02_心地よい暖炉.mp3" min="0" max="100" value="0">
                <span class="zsp-volume-value">0%</span>
              </div>
            </div>
          </div>
          <div class="zsp-ambient-item">
            <div class="zsp-ambient-info">
              <span>とどろく雷鳴</span>
              <div class="zsp-slider-container">
                <input type="range" class="zsp-slider" data-sound="03_とどろく雷鳴.wav" min="0" max="100" value="0">
                <span class="zsp-volume-value">0%</span>
              </div>
            </div>
          </div>
          <div class="zsp-ambient-item">
            <div class="zsp-ambient-info">
              <span>雨と雷鳴</span>
              <div class="zsp-slider-container">
                <input type="range" class="zsp-slider" data-sound="04_雨と雷鳴.mp3" min="0" max="100" value="0">
                <span class="zsp-volume-value">0%</span>
              </div>
            </div>
          </div>
          <div class="zsp-ambient-item">
            <div class="zsp-ambient-info">
              <span>夜の森のコオロギ</span>
              <div class="zsp-slider-container">
                <input type="range" class="zsp-slider" data-sound="05_夜の森のコオロギ.mp3" min="0" max="100" value="0">
                <span class="zsp-volume-value">0%</span>
              </div>
            </div>
          </div>
          <div class="zsp-ambient-item">
            <div class="zsp-ambient-info">
              <span>穏やかな風</span>
              <div class="zsp-slider-container">
                <input type="range" class="zsp-slider" data-sound="06_穏やかな風.mp3" min="0" max="100" value="0">
                <span class="zsp-volume-value">0%</span>
              </div>
            </div>
          </div>
        </div>
        
        <div class="zsp-control-group">
          <h3>集中音</h3>
          <div class="zsp-ambient-item">
            <div class="zsp-ambient-info">
              <span>ホワイトノイズ</span>
              <div class="zsp-slider-container">
                <input type="range" class="zsp-slider" data-sound="10_ホワイトノイズ.wav" min="0" max="100" value="0">
                <span class="zsp-volume-value">0%</span>
              </div>
            </div>
          </div>
          <div class="zsp-ambient-item">
            <div class="zsp-ambient-info">
              <span>ピンクノイズ</span>
              <div class="zsp-slider-container">
                <input type="range" class="zsp-slider" data-sound="11_ピンクノイズ.wav" min="0" max="100" value="0">
                <span class="zsp-volume-value">0%</span>
              </div>
            </div>
          </div>
          <div class="zsp-ambient-item">
            <div class="zsp-ambient-info">
              <span>キーボードタイピング</span>
              <div class="zsp-slider-container">
                <input type="range" class="zsp-slider" data-sound="12_キーボードタイピング.mp3" min="0" max="100" value="0">
                <span class="zsp-volume-value">0%</span>
              </div>
            </div>
          </div>
        </div>
      </div>
    `;
  }

  setupEventListeners() {
    // マウス近接検知（0.1秒以上ホバーでのみ開く）
    window.addEventListener('mousemove', (e) => {
      if (this.lockState) return;
      const cx = window.innerWidth / 2;
      const dx = Math.abs(e.clientX - cx);
      const nearTop = e.clientY < 110;
      const nearCenter = dx < 180;
      const onIsland = this.islandElement.contains(e.target);

      // ホバー判定
      if ((nearTop && nearCenter) || onIsland) {
        if (!this.hoverActive && !this.hoverTimer) {
          this.hoverTimer = setTimeout(() => {
            this.hoverActive = true;
            this.setActive(true);
          }, 200); // 0.2秒
        }
      } else {
        // マウスがエリア外に出た時は、mouseleaveタイマーに任せる
        // 即座に閉じないようにする
        if (this.hoverTimer) {
          clearTimeout(this.hoverTimer);
          this.hoverTimer = null;
        }
        this.hoverActive = false;
        // this.setActive(false); を削除して、mouseleaveの遅延処理に任せる
      }
    });

    // アイランドにマウスが入った時
    this.islandElement.addEventListener('mouseenter', () => {
      // mouseleaveタイマーをキャンセル
      if (this.mouseleaveTimer) {
        clearTimeout(this.mouseleaveTimer);
        this.mouseleaveTimer = null;
      }
    });

    // アイランドから出たら閉じる
    this.islandElement.addEventListener('mouseleave', () => {
      if (!this.lockState) {
        // 既存のタイマーをクリア
        if (this.mouseleaveTimer) {
          clearTimeout(this.mouseleaveTimer);
          this.mouseleaveTimer = null;
        }
        
        // 0.1秒後に閉じる
        this.mouseleaveTimer = setTimeout(() => {
          if (this.hoverTimer) {
            clearTimeout(this.hoverTimer);
            this.hoverTimer = null;
          }
          this.hoverActive = false;
          this.setActive(false);
          this.mouseleaveTimer = null;
        }, 200);
      }
    });


    // 外側クリックで閉じる
    document.addEventListener('click', (e) => {
      if (!this.islandElement.contains(e.target)) {
        if (this.hoverTimer) {
          clearTimeout(this.hoverTimer);
          this.hoverTimer = null;
        }
        this.hoverActive = false;
        this.setActive(false);
      }
    });

    // Escで閉じる
    document.addEventListener('keydown', (e) => {
      if (e.key === 'Escape') this.setActive(false);
    });

    // パネル切り替え
    this.islandElement.addEventListener('click', (e) => {
      if (e.target.classList.contains('zsp-nav-btn')) {
        this.switchPanel(e.target.dataset.panel);
      }
    });

    // 各種コントロールのイベントリスナー
    this.setupControlListeners();

    // ログイン状態の変化を監視
    this._onStorageChanged = (changes, areaName) => {
      if (areaName === 'local' && Object.prototype.hasOwnProperty.call(changes, 'jwt')) {
        this.isLoggedIn = !!changes.jwt.newValue;
        this.updateLoginUI();
      }
    };
    chrome.storage.onChanged.addListener(this._onStorageChanged);
  }

  setupControlListeners() {
    const island = this.islandElement;

    // タイマーモード切り替え
    const timerModeButtons = island.querySelectorAll('.zsp-timer-mode-btn');
    timerModeButtons.forEach(button => {
      button.addEventListener('click', (e) => {
        this.timerMode = e.target.dataset.mode;
        this.setActiveTimerMode(this.timerMode);
        this.updateTimerDisplay();
        chrome.storage.sync.set({ timerMode: this.timerMode });
      });
    });

    // タイマー設定を読み込み
    chrome.storage.sync.get('timerMode', (result) => {
      if (result.timerMode) {
        this.timerMode = result.timerMode;
        this.setActiveTimerMode(this.timerMode);
      }
    });

    // タイマーボタン
    island.querySelector('#zsp-timer-start')?.addEventListener('click', () => this.startTimer());
    island.querySelector('#zsp-timer-stop')?.addEventListener('click', () => this.stopTimer());
    island.querySelector('#zsp-timer-skip')?.addEventListener('click', () => this.skipTimer());

    // ログイン導線ボタン
    island.querySelector('#zsp-go-login')?.addEventListener('click', () => {
      try { chrome.runtime.sendMessage({ action: 'OPEN_ACCOUNT_TAB' }); } catch (_) {}
    });

    // 音楽関連
    const musicVolumeSlider = island.querySelector('#zsp-music-volume');
    const musicVolumeValue = island.querySelector('#zsp-music-volume-value');
    const volumeControl = island.querySelector('.zsp-volume-control');
    const volumeSliderContainer = island.querySelector('.zsp-volume-slider-container');

    // 音量スライダーのイベント
    musicVolumeSlider?.addEventListener('input', (e) => {
      this.musicState.volume = parseInt(e.target.value);
      musicVolumeValue.textContent = `${e.target.value}%`;
      this.updateMusicDisplay(); // SVGアイコンを更新
      this.updateMusicVolume();
    });

    // 音量コントロールのホバー管理を改善
    let volumeHoverTimeout;
    
    const showVolumeSlider = () => {
      if (volumeHoverTimeout) {
        clearTimeout(volumeHoverTimeout);
      }
      volumeSliderContainer.style.opacity = '1';
      volumeSliderContainer.style.pointerEvents = 'auto';
    };

    const hideVolumeSlider = () => {
      volumeHoverTimeout = setTimeout(() => {
        volumeSliderContainer.style.opacity = '0';
        volumeSliderContainer.style.pointerEvents = 'none';
      }, 300); // 300ms の遅延で非表示
    };

    // ボタンとスライダーコンテナの両方でホバーイベントを管理
    volumeControl?.addEventListener('mouseenter', showVolumeSlider);
    volumeControl?.addEventListener('mouseleave', hideVolumeSlider);
    volumeSliderContainer?.addEventListener('mouseenter', showVolumeSlider);
    volumeSliderContainer?.addEventListener('mouseleave', hideVolumeSlider);

    // 音楽コントロール
    island.querySelector('#zsp-music-prev')?.addEventListener('click', () => this.previousTrack());
    island.querySelector('#zsp-music-play')?.addEventListener('click', () => this.toggleMusic());
    island.querySelector('#zsp-music-next')?.addEventListener('click', () => this.nextTrack());

    // 環境音スライダー
    const ambientSliders = island.querySelectorAll('.zsp-slider[data-sound]');
    ambientSliders.forEach(slider => {
      const volumeValue = slider.parentElement.querySelector('.zsp-volume-value');
      
      slider.addEventListener('input', (e) => {
        const volume = parseInt(e.target.value);
        const soundFile = e.target.dataset.sound;
        volumeValue.textContent = `${volume}%`;
        this.setAmbientVolume(soundFile, volume);
      });
    });

    // 環境音全停止
    island.querySelector('#zsp-ambient-stop-all')?.addEventListener('click', () => this.stopAllAmbient());

    island.querySelector('#zsp-open-dashboard')?.addEventListener('click', () => this.openDashboard());

    // ミュージック タブ (Lo‑Fi / Spotify)
    const musicTabButtons = island.querySelectorAll('.zsp-music-mode-btn[data-music]');
    const lofiPane = island.querySelector('.zsp-music-pane[data-pane="lofi"]');
    const spotifyPane = island.querySelector('.zsp-music-pane[data-pane="spotify"]');
    const setMusicTab = (key) => {
      musicTabButtons.forEach(b => b.classList.toggle('active', b.dataset.music === key));
      lofiPane.style.display = key === 'lofi' ? 'block' : 'none';
      spotifyPane.style.display = key === 'spotify' ? 'block' : 'none';
      chrome.storage.sync.set({ musicService: key });
      if (key === 'spotify') this.updateSpotifyUI();
    };
    musicTabButtons.forEach(btn => btn.addEventListener('click', () => setMusicTab(btn.dataset.music)));

    // 初期化: 前回の選択を復元
    chrome.storage.sync.get('musicService', (res) => setMusicTab(res.musicService || 'lofi'));

    // Spotify controls (minimal)
    const spPane = island.querySelector('.zsp-music-pane[data-pane="spotify"]');
    const spConnectBtn = spPane?.querySelector('#zsp-spotify-connect');
    const spPlayBtn = spPane?.querySelector('#zsp-spotify-play');
    const spNextBtn = spPane?.querySelector('#zsp-spotify-next');
    const spPrevBtn = spPane?.querySelector('#zsp-spotify-prev');
    const spVol = spPane?.querySelector('#zsp-spotify-volume');
    const spVolValue = spPane?.querySelector('#zsp-spotify-volume-value');
    const spVolControl = spPane?.querySelector('.zsp-volume-control');
    const spVolContainer = spPane?.querySelector('.zsp-volume-slider-container');

    spConnectBtn?.addEventListener('click', async () => {
      const btnText = spConnectBtn.textContent;
      spConnectBtn.disabled = true;
      
      try {
        if (btnText === '設定へ') {
          // background経由でポップアップを開く
          chrome.runtime.sendMessage({ action: 'OPEN_SETTINGS_TAB' });
        } else if (btnText === '接続' || btnText === '再試行') {
          // Spotify認証を開始
          spConnectBtn.textContent = '接続中...';
          await chrome.runtime.sendMessage({ action: 'SPOTIFY_AUTH' });
        }
      } catch (e) {
        console.warn('Spotify connection failed:', e);
      } finally {
        spConnectBtn.disabled = false;
        this.updateSpotifyUI();
      }
    });
    spPlayBtn?.addEventListener('click', async () => {
      try { await chrome.runtime.sendMessage({ action: 'SPOTIFY_TOGGLE' }); } catch (e) {}
      setTimeout(() => this.updateSpotifyUI(), 400);
    });
    spNextBtn?.addEventListener('click', async () => {
      try { await chrome.runtime.sendMessage({ action: 'SPOTIFY_NEXT' }); } catch (e) {}
      setTimeout(() => this.updateSpotifyUI(), 400);
    });
    spPrevBtn?.addEventListener('click', async () => {
      try { await chrome.runtime.sendMessage({ action: 'SPOTIFY_PREV' }); } catch (e) {}
      setTimeout(() => this.updateSpotifyUI(), 400);
    });
    let spVolTimer;
    spVol?.addEventListener('input', () => {
      const v = parseInt(spVol.value || '0', 10);
      if (spVolValue) spVolValue.textContent = `${v}%`;
      clearTimeout(spVolTimer);
      spVolTimer = setTimeout(async () => {
        try { await chrome.runtime.sendMessage({ action: 'SPOTIFY_VOLUME', percent: v }); } catch (e) {}
      }, 150);
    });

    // Spotify volume popover show/hide (Lo‑Fi style)
    let spVolHoverTimer;
    const spShowVol = () => {
      if (!spVolContainer) return;
      if (spVolHoverTimer) clearTimeout(spVolHoverTimer);
      spVolContainer.style.opacity = '1';
      spVolContainer.style.pointerEvents = 'auto';
    };
    const spHideVol = () => {
      if (!spVolContainer) return;
      spVolHoverTimer = setTimeout(() => {
        spVolContainer.style.opacity = '0';
        spVolContainer.style.pointerEvents = 'none';
      }, 300);
    };
    spVolControl?.addEventListener('mouseenter', spShowVol);
    spVolControl?.addEventListener('mouseleave', spHideVol);
    spVolContainer?.addEventListener('mouseenter', spShowVol);
    spVolContainer?.addEventListener('mouseleave', spHideVol);

    // No back button (tab switching remains at top)
  }

  setActive(active) {
    if (this.isActive === active) return;
    
    this.isActive = active;
    this.islandElement.setAttribute('data-active', active ? 'true' : 'false');
    this.islandElement.querySelector('.zsp-island-panel').setAttribute('aria-hidden', active ? 'false' : 'true');
    
    if (active) {
      this.updateStatus();
      this.startPolling();
      this.stopCompactPolling();
    } else {
      this.stopPolling();
      // 維持: 最小化中もSpotifyの曲変更を検知する軽量ポーリング
      this.startCompactPolling();
    }
  }

  // 表示中のモードボタンだけを更新（状態フェッチ時のUI整合用）
  updateTimerModeButtons(mode) {
    this.islandElement.querySelectorAll('.zsp-timer-mode-btn').forEach(btn => {
      btn.classList.toggle('active', btn.dataset.mode === mode);
    });
  }

  // 現在モードに応じたデフォルト表示（未アクティブ時の上書き防止）
  updateTimerDisplay() {
    const display = this.islandElement.querySelector('#zsp-timer-display');
    const label = this.islandElement.querySelector('#zsp-timer-label');
    if (!display || !label) return;
    if (this.timerMode === 'manual') {
      display.textContent = '00:00';
      label.textContent = '集中時間';
    } else {
      const defaultTime = this.timerMode === 'pomodoro' ? '25:00' : '45:00';
      display.textContent = defaultTime;
      label.textContent = '集中時間';
    }
  }

  switchPanel(panelName) {
    // ナビゲーションボタンの更新
    this.islandElement.querySelectorAll('.zsp-nav-btn').forEach(btn => {
      btn.classList.toggle('active', btn.dataset.panel === panelName);
    });

    // パネルコンテンツの切り替え
    this.islandElement.querySelectorAll('.zsp-panel-content').forEach(panel => {
      panel.classList.toggle('active', panel.dataset.panel === panelName);
    });

    this.currentPanel = panelName;
    
    // パネル切り替え時の個別処理
    if (panelName === 'main') {
      this.fetchStatus();
      this.updateMusicDisplay();
    }
  }

  // タイマーモード関連メソッド
  setActiveTimerMode(mode) {
    this.islandElement.querySelectorAll('.zsp-timer-mode-btn').forEach(btn => {
      btn.classList.toggle('active', btn.dataset.mode === mode);
    });
    this.fetchStatus();
  }

  // タイマー関連メソッド (background通信)
  async startTimer() {
    if (!this.isLoggedIn) {
      this.showNotification('ログインが必要です', 'タイマー機能を使用するには、拡張機能アイコンからログインしてください。');
      return;
    }

    if (this.timerMode === 'manual') {
      chrome.runtime.sendMessage({ type: 'studyStart' }, (res) => {
        if (res && res.ok) this.fetchStatus();
      });
    } else {
      chrome.runtime.sendMessage({ type: 'advancedTimerStart', mode: this.timerMode }, (res) => {
        if (res && res.ok) this.fetchStatus();
      });
    }
  }

  stopTimer() {
    if (this.timerMode === 'manual') {
      chrome.runtime.sendMessage({ type: 'studyStop' }, () => this.fetchStatus());
    } else {
      chrome.runtime.sendMessage({ type: 'advancedTimerStop' }, () => this.fetchStatus());
    }
  }

  skipTimer() {
    if (this.timerMode === 'manual') return;
    
    const state = this.timerState;
    if (state.phase === 'focus') {
      chrome.runtime.sendMessage({ type: 'advancedTimerSkipToBreak' }, () => this.fetchStatus());
    } else {
      chrome.runtime.sendMessage({ type: 'advancedTimerSkipBreak' }, () => this.fetchStatus());
    }
  }

  // 手動・高度の両方を取得し、実際にアクティブな方を表示
  fetchStatus() {
    let manualState = null;
    let advState = null;
    let got = 0;
    const finalize = () => {
      if (got < 2) return;
      const aActive = !!(advState && advState.active);
      const mActive = !!(manualState && manualState.active);

      if (aActive) {
        this.timerMode = advState.mode || 'pomodoro';
        this.timerState = advState;
        this.updateTimerModeButtons(this.timerMode);
        this.updateAdvancedTimerUI();
        // 表示アンカーを同期
        this._displayAnchor = {
          ts: Date.now(),
          manualElapsedMs: 0,
          advRemainSec: advState.remainSec || 0
        };
        return;
      }
      if (mActive) {
        this.timerMode = 'manual';
        this.timerState = { ...manualState, mode: 'manual' };
        this.updateTimerModeButtons('manual');
        this.updateFocusUI();
        // 表示アンカーを同期
        this._displayAnchor = {
          ts: Date.now(),
          manualElapsedMs: manualState.currentElapsedMs || 0,
          advRemainSec: 0
        };
        return;
      }
      // どちらも非アクティブ
      this.timerState = { active: false };
      if (this.timerMode === 'manual') {
        this.updateFocusUI();
      } else {
        this.updateAdvancedTimerUI();
      }
      this.updateCompactStatus();
      // 非アクティブなのでアンカーをリセット
      this._displayAnchor = { ts: 0, manualElapsedMs: 0, advRemainSec: 0 };
    };

    chrome.runtime.sendMessage({ type: 'studyStatus' }, (state) => {
      manualState = state || { active: false };
      got += 1;
      finalize();
    });
    chrome.runtime.sendMessage({ type: 'advancedTimerStatus' }, (state) => {
      advState = state || { active: false };
      got += 1;
      finalize();
    });
  }

  updateFocusUI() {
    const state = this.timerState;
    const display = this.islandElement.querySelector('#zsp-timer-display');
    const label = this.islandElement.querySelector('#zsp-timer-label');
    const startBtn = this.islandElement.querySelector('#zsp-timer-start');
    const stopBtn = this.islandElement.querySelector('#zsp-timer-stop');
    const skipBtn = this.islandElement.querySelector('#zsp-timer-skip');

    if (!state || !state.active) {
      display.textContent = '00:00';
      label.textContent = '集中時間';
      startBtn.style.display = 'inline-block';
      stopBtn.style.display = 'none';
      skipBtn.style.display = 'none';
    } else {
      const elapsedMs = state.currentElapsedMs || 0;
      const minutes = Math.floor(elapsedMs / 60000);
      const seconds = Math.floor((elapsedMs % 60000) / 1000);
      display.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
      label.textContent = '学習中';
      startBtn.style.display = 'none';
      stopBtn.style.display = 'inline-block';
      skipBtn.style.display = 'none';
    }
    this.updateCompactStatus();
  }

  updateAdvancedTimerUI() {
    const state = this.timerState;
    const display = this.islandElement.querySelector('#zsp-timer-display');
    const label = this.islandElement.querySelector('#zsp-timer-label');
    const startBtn = this.islandElement.querySelector('#zsp-timer-start');
    const stopBtn = this.islandElement.querySelector('#zsp-timer-stop');
    const skipBtn = this.islandElement.querySelector('#zsp-timer-skip');

    if (!state || !state.active) {
      const defaultTime = this.timerMode === 'pomodoro' ? '25:00' : '45:00';
      display.textContent = defaultTime;
      label.textContent = '集中時間';
      startBtn.style.display = 'inline-block';
      stopBtn.style.display = 'none';
      skipBtn.style.display = 'none';
    } else {
      const remainSec = state.remainSec || 0;
      const minutes = Math.floor(remainSec / 60);
      const seconds = remainSec % 60;
      display.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
      
      const phaseText = state.phase === 'focus' ? '集中' : (state.phase === 'short_break' ? '短休憩' : '長休憩');
      label.textContent = `${phaseText} - ${state.cycle}`;
      
      startBtn.style.display = 'none';
      stopBtn.style.display = 'inline-block';
      skipBtn.style.display = 'inline-block';
      skipBtn.textContent = state.phase === 'focus' ? '休憩へ' : '集中へ';
    }
    this.updateCompactStatus();
  }

  // 音楽関連メソッド
  updateMusicDisplay() {
    const titleEl = this.islandElement.querySelector('#zsp-track-title');
    const playBtn = this.islandElement.querySelector('#zsp-music-play');
    const volumeSvg = this.islandElement.querySelector('#zsp-volume-svg');

    if (this.musicState.tracks.length > 0) {
      const currentTrack = this.musicState.tracks[this.musicState.currentTrackIndex];
      titleEl.textContent = `${currentTrack.title} (${this.musicState.currentTrackIndex + 1}/${this.musicState.tracks.length})`;
    }

    if (this.musicState.isPlaying) {
      playBtn.innerHTML = '⏸';
      playBtn.title = '一時停止';
    } else {
      playBtn.innerHTML = '▶';
      playBtn.title = '再生';
    }

    // 音量に応じてSVGアイコンを変更
    if (volumeSvg) {
      if (this.musicState.volume === 0) {
        // ミュート
        volumeSvg.innerHTML = '<path d="M16.5 12c0-1.77-1.02-3.29-2.5-4.03v2.21l2.45 2.45c.03-.2.05-.41.05-.63zm2.5 0c0 .94-.2 1.82-.54 2.64l1.51 1.51C20.63 14.91 21 13.5 21 12c0-4.28-2.99-7.86-7-8.77v2.06c2.89.86 5 3.54 5 6.71zM4.27 3L3 4.27 7.73 9H3v6h4l5 5v-6.73l4.25 4.25c-.67.52-1.42.93-2.25 1.18v2.06c1.38-.31 2.63-.95 3.69-1.81L19.73 21 21 19.73l-9-9L4.27 3zM12 4L9.91 6.09 12 8.18V4z"/>';
      } else if (this.musicState.volume < 30) {
        // 低音量
        volumeSvg.innerHTML = '<path d="M7 9v6h4l5 5V4l-5 5H7z"/>';
      } else if (this.musicState.volume < 70) {
        // 中音量
        volumeSvg.innerHTML = '<path d="M3 9v6h4l5 5V4L7 9H3zm13.5 3c0-1.77-1.02-3.29-2.5-4.03v8.05c1.48-.73 2.5-2.25 2.5-4.02z"/>';
      } else {
        // 高音量
        volumeSvg.innerHTML = '<path d="M3 9v6h4l5 5V4L7 9H3zm13.5 3c0-1.77-1.02-3.29-2.5-4.03v8.05c1.48-.73 2.5-2.25 2.5-4.02zM14 3.23v2.06c2.89.86 5 3.54 5 6.71s-2.11 5.85-5 6.71v2.06c4.01-.91 7-4.49 7-8.77s-2.99-7.86-7-8.77z"/>';
      }
    }
  }

  toggleMusic() {
    if (this.musicState.isPlaying) {
      this.pauseMusic();
    } else {
      this.playMusic();
    }
  }

  playMusic() {
    if (this.musicState.tracks.length === 0) return;

    const currentTrack = this.musicState.tracks[this.musicState.currentTrackIndex];
    
    // UIを即時更新
    this.musicState.isPlaying = true;
    this.updateMusicDisplay();
    this.updateCompactStatus();

    chrome.runtime.sendMessage({
      type: 'playLofi',
      fileName: currentTrack.file,
      volume: this.musicState.volume,
      trackIndex: this.musicState.currentTrackIndex
    });
  }

  pauseMusic() {
    chrome.runtime.sendMessage({ type: 'pauseLofi' });
    this.musicState.isPlaying = false;
    this.updateMusicDisplay();
    this.updateCompactStatus();
  }

  stopMusic() {
    chrome.runtime.sendMessage({ type: 'stopLofi' });
    this.musicState.isPlaying = false;
    this.updateMusicDisplay();
    this.updateCompactStatus();
  }

  nextTrack() {
    if (this.musicState.tracks.length === 0) return;
    
    this.musicState.currentTrackIndex = (this.musicState.currentTrackIndex + 1) % this.musicState.tracks.length;
    this.playMusic();
  }

  previousTrack() {
    if (this.musicState.tracks.length === 0) return;
    
    this.musicState.currentTrackIndex = this.musicState.currentTrackIndex === 0 
      ? this.musicState.tracks.length - 1 
      : this.musicState.currentTrackIndex - 1;
    this.playMusic();
  }

  updateMusicVolume() {
    chrome.runtime.sendMessage({
      type: 'setLofiVolume',
      volume: this.musicState.volume
    });
  }

  // 環境音関連メソッド
  setAmbientVolume(soundFile, volume) {
    if (volume === 0) {
      chrome.runtime.sendMessage({
        type: 'stopAmbient',
        fileName: soundFile
      });
    } else {
      chrome.runtime.sendMessage({
        type: 'playAmbient',
        fileName: soundFile,
        volume: volume
      });
    }
  }

  stopAllAmbient() {
    chrome.runtime.sendMessage({ type: 'stopAllAmbient' });
    
    // 全てのスライダーを0にリセット
    const ambientSliders = this.islandElement.querySelectorAll('.zsp-slider[data-sound]');
    ambientSliders.forEach(slider => {
      slider.value = 0;
      const volumeValue = slider.parentElement.querySelector('.zsp-volume-value');
      volumeValue.textContent = '0%';
    });
  }

  openDashboard() {
    window.open('https://zsp.yoima.com/dashboard/', '_blank');
  }

  // 状態更新メソッド
  updateStatus() {
    this.checkLoginStatus();
    this.fetchStatus();
  }

  async updateSpotifyUI() {
    try {
      const resp = await chrome.runtime.sendMessage({ action: 'SPOTIFY_STATUS' });
      const el = this.islandElement.querySelector('#zsp-spotify-title');
      const playBtn = this.islandElement.querySelector('#zsp-spotify-play');
      const connectBtn = this.islandElement.querySelector('#zsp-spotify-connect');
      const coverEl = this.islandElement.querySelector('#zsp-spotify-cover');
      const volEl = this.islandElement.querySelector('#zsp-spotify-volume');
      const volVal = this.islandElement.querySelector('#zsp-spotify-volume-value');
      if (!el || !playBtn || !connectBtn) return;
      
      // カバー画像のリセット関数
      const resetCover = () => {
        if (coverEl) {
          coverEl.src = '';
          coverEl.className = 'zsp-spotify-cover hidden';
          coverEl.style.display = 'none';
        }
      };
      
      // カバー画像をプレースホルダーに設定
      const setPlaceholderCover = () => {
        if (coverEl) {
          coverEl.src = chrome.runtime.getURL('images/music.png');
          coverEl.className = 'zsp-spotify-cover';
          coverEl.style.display = 'block';
        }
      };
      
      if (!resp || !resp.success) {
        el.textContent = 'Spotify接続でエラーが発生しました';
        connectBtn.style.display = 'inline-flex';
        connectBtn.textContent = '再試行';
        playBtn.disabled = true;
        resetCover();
        this.spotifyCompact = { isPlaying: false, title: '', artist: '', cover: '' };
        return;
      }
      
      // クライアントIDが未設定の場合
      if (!resp.hasClientId) {
        el.textContent = '設定でSpotifyクライアントIDを入力してください';
        connectBtn.style.display = 'inline-flex';
        connectBtn.textContent = '設定へ';
        playBtn.disabled = true;
        resetCover();
        this.spotifyCompact = { isPlaying: false, title: '', artist: '', cover: '' };
        return;
      }
      
      // クライアントIDは設定済みだが未認証の場合
      if (!resp.authed) {
        el.textContent = 'Spotifyとの連携が必要です。接続してください';
        connectBtn.style.display = 'inline-flex';
        connectBtn.textContent = '接続';
        playBtn.disabled = true;
        resetCover();
        this.spotifyCompact = { isPlaying: false, title: '', artist: '', cover: '' };
        return;
      }
      
      // 認証済みの場合
      connectBtn.style.display = 'none';
      const p = resp.player;
      if (p && p.item) {
        const title = p.item.name || '';
        const artist = (p.item.artists || []).map(a => a.name).join(', ');
        el.textContent = `${title} — ${artist}`;
        playBtn.textContent = p.is_playing ? '⏸' : '▶';
        playBtn.disabled = false;
        // cover
        const images = p.item.album?.images || [];
        const smallest = images[images.length - 1] || images[0];
        if (coverEl) {
          if (smallest?.url) {
            coverEl.className = 'zsp-spotify-cover';
            coverEl.style.display = 'block';
            coverEl.src = smallest.url;
            // 画像読み込みエラー時の処理
            coverEl.onerror = () => {
              setPlaceholderCover();
            };
          } else {
            setPlaceholderCover();
          }
        }
        this.spotifyCompact = {
          isPlaying: !!p.is_playing,
          title,
          artist,
          cover: smallest?.url || ''
        };
        // volume
        const v = p.device?.volume_percent;
        if (typeof v === 'number' && volEl && volVal) {
          volEl.value = String(v);
          volVal.textContent = `${v}%`;
        }
      } else {
        el.textContent = 'アクティブなSpotifyデバイスが見つかりません';
        playBtn.disabled = false;
        if (coverEl) {
          coverEl.src = '';
          coverEl.className = 'zsp-spotify-cover hidden';
          coverEl.style.display = 'none';
        }
        this.spotifyCompact = { isPlaying: false, title: '', artist: '', cover: '' };
      }
    } catch (e) {
      const el = this.islandElement.querySelector('#zsp-spotify-title');
      const connectBtn = this.islandElement.querySelector('#zsp-spotify-connect');
      const coverEl = this.islandElement.querySelector('#zsp-spotify-cover');
      if (el) el.textContent = 'Spotifyサービスに接続できません';
      if (connectBtn) {
        connectBtn.style.display = 'inline-flex';
        connectBtn.textContent = '再試行';
      }
      if (coverEl) {
        coverEl.src = '';
        coverEl.className = 'zsp-spotify-cover hidden';
        coverEl.style.display = 'none';
      }
    }
    this.updateCompactStatus();
  }

  checkLoginStatus() {
    chrome.storage.local.get('jwt', (result) => {
      this.isLoggedIn = !!result.jwt;
      this.updateLoginUI();
    });
  }

  // ログイン状態に基づいてタイマーUIを更新
  updateLoginUI() {
    const overlay = this.islandElement.querySelector('#zsp-login-overlay');
    const group = overlay?.closest('.zsp-control-group');
    const startBtn = this.islandElement.querySelector('#zsp-timer-start');
    const stopBtn = this.islandElement.querySelector('#zsp-timer-stop');
    const skipBtn = this.islandElement.querySelector('#zsp-timer-skip');
    const modeBtns = this.islandElement.querySelectorAll('.zsp-timer-mode-btn');

    if (!overlay || !group) return;

    if (this.isLoggedIn) {
      overlay.style.display = 'none';
      group.classList.remove('zsp-login-blocked');
      startBtn?.removeAttribute('disabled');
      stopBtn?.removeAttribute('disabled');
      skipBtn?.removeAttribute('disabled');
      modeBtns.forEach(b => b.removeAttribute('disabled'));
    } else {
      overlay.style.display = 'flex';
      group.classList.add('zsp-login-blocked');
      startBtn?.setAttribute('disabled', 'true');
      stopBtn?.setAttribute('disabled', 'true');
      skipBtn?.setAttribute('disabled', 'true');
      modeBtns.forEach(b => b.setAttribute('disabled', 'true'));
    }
  }

  updateCompactStatus() {
    const compactEl = this.islandElement.querySelector('.zsp-island-compact');
    const statusDot = this.islandElement.querySelector('.zsp-status-dot');
    const state = this.timerState;

    if (state && state.active) {
      let timeStr = '00:00';
      if (this.timerMode === 'manual') {
        const elapsedMs = state.currentElapsedMs || 0;
        const minutes = Math.floor(elapsedMs / 60000);
        const seconds = Math.floor((elapsedMs % 60000) / 1000);
        timeStr = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
      } else {
        const remainSec = state.remainSec || 0;
        const minutes = Math.floor(remainSec / 60);
        const seconds = remainSec % 60;
        timeStr = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
      }
      
      // Spotifyが再生中ならカバー画像も表示
      const spotifyPlaying = this.spotifyCompact && this.spotifyCompact.isPlaying;
      const spotifyCover = spotifyPlaying ? this.spotifyCompact.cover : '';
      
      compactEl.innerHTML = `
        <div class="zsp-status-indicator">
          <div class="zsp-status-dot"></div>
          ${spotifyPlaying && spotifyCover ? `<img class="zsp-compact-cover" src="${spotifyCover}" alt="cover" onerror="this.src='${chrome.runtime.getURL('images/music.png')}'" />` : ''}
          ${timeStr}
        </div>
      `;
      statusDot?.classList?.remove('inactive');
    } else if (this.musicState.isPlaying) {
      const currentTrack = this.musicState.tracks[this.musicState.currentTrackIndex];
      const title = currentTrack ? currentTrack.title : '再生中';
      compactEl.innerHTML = `
        <div class="zsp-status-indicator">
          <div class="zsp-status-dot"></div>
          <span aria-hidden="true">♪</span>
          <span class="zsp-compact-title">
            <span class="zsp-marquee">
              <span class="zsp-marquee-track">
                <span class="zsp-marquee-text">${title}</span>
              </span>
            </span>
          </span>
        </div>
      `;
      statusDot?.classList?.remove('inactive');
      // マルチコピーのトラック幅に基づき、必要時のみスクロールを有効化
      this.setupCompactMarquee();
    } else if (this.spotifyCompact && this.spotifyCompact.isPlaying) {
      const title = this.spotifyCompact.title || '再生中';
      const artist = this.spotifyCompact.artist || '';
      const cover = this.spotifyCompact.cover || '';
      const text = artist ? `${title} — ${artist}` : title;
      compactEl.innerHTML = `
        <div class="zsp-status-indicator">
          <div class="zsp-status-dot"></div>
          ${cover ? `<img class="zsp-compact-cover" src="${cover}" alt="cover" onerror="this.src='${chrome.runtime.getURL('images/music.png')}'" />` : `<img class="zsp-compact-cover" src="${chrome.runtime.getURL('images/music.png')}" alt="cover" />`}
          <span class="zsp-compact-title">
            <span class="zsp-marquee">
              <span class="zsp-marquee-track">
                <span class="zsp-marquee-text">${text}</span>
              </span>
            </span>
          </span>
        </div>
      `;
      statusDot?.classList?.remove('inactive');
      this.setupCompactMarquee();
    } else {
      compactEl.innerHTML = `
        <div class="zsp-status-indicator" aria-label="Inactive">
          <div class="zsp-status-dot inactive"></div>
        </div>
      `;
      statusDot?.classList?.add('inactive');
    }
  }

  // コンパクト表示の曲名が容器幅を超える場合のみスクロール有効化
  setupCompactMarquee() {
    try {
      const container = this.islandElement.querySelector('.zsp-compact-title');
      const track = this.islandElement.querySelector('.zsp-marquee-track');
      if (!container || !track) return;
      // container幅よりコンテンツ半分（=1コピー）幅が大きければスクロール
      const texts = track.querySelectorAll('.zsp-marquee-text');
      if (texts.length < 1) return;
      // 1つ目のコピーの幅を測定
      const singleWidth = texts[0].scrollWidth;
      const needScroll = singleWidth > container.clientWidth;
      // 必要時のみ2つ目を追加、不要なら重複を除去
      if (needScroll) {
        if (texts.length === 1) {
          const clone = texts[0].cloneNode(true);
          track.appendChild(clone);
        }
      } else {
        for (let i = texts.length - 1; i >= 1; i--) {
          texts[i].remove();
        }
      }
      container.classList.toggle('too-long', needScroll);
    } catch (e) {
      // 計測失敗時は何もしない
    }
  }

  showNotification(title, message) {
    if ('Notification' in window && Notification.permission === 'granted') {
      new Notification(title, {
        body: message,
        icon: chrome.runtime.getURL('images/notification_icon.png')
      });
    }
  }

  loadSettings() {
    chrome.storage.sync.get(['timerMode'], (result) => {
      if (result.timerMode) {
        this.timerMode = result.timerMode;
        this.setActiveTimerMode(this.timerMode);
      }
      
      this.updateMusicDisplay();
      this.fetchStatus();
    });
    this.checkLoginStatus();
  }

  // 定期更新
  startPolling() {
    if (this.pollingInterval) return;
    this.pollingInterval = setInterval(() => {
      // 展開状態に関わらず、状態を取得してUI/コンパクト表示を更新
      this.fetchStatus();
      this.updateSpotifyUI?.();
      this.checkLoginStatus();
    }, 1000);
  }

  stopPolling() {
    if (this.pollingInterval) {
      clearInterval(this.pollingInterval);
      this.pollingInterval = null;
    }
  }

  // 最小化時の軽量ポーリング（Spotifyのみ 5秒間隔）
  startCompactPolling() {
    if (this.compactPollingInterval) return;
    this.compactPollingInterval = setInterval(() => {
      this.updateSpotifyUI?.();
    }, 5000);
  }

  stopCompactPolling() {
    if (this.compactPollingInterval) {
      clearInterval(this.compactPollingInterval);
      this.compactPollingInterval = null;
    }
  }

  // ローカルの1秒チックでUIだけ更新（バックグラウンド実状態はfetchStatusで同期）
  startDisplayTicker() {
    if (this.displayTicker) return;
    this.displayTicker = setInterval(() => {
      if (!this.timerState || !this.timerState.active) return;
      const now = Date.now();
      if (this.timerMode === 'manual') {
        const base = this._displayAnchor.manualElapsedMs || 0;
        const elapsed = base + (now - (this._displayAnchor.ts || now));
        this.timerState.currentElapsedMs = Math.max(0, elapsed);
        this.updateFocusUI();
      } else {
        const baseRemain = this._displayAnchor.advRemainSec || 0;
        const passed = Math.floor((now - (this._displayAnchor.ts || now)) / 1000);
        const remain = Math.max(0, baseRemain - passed);
        this.timerState.remainSec = remain;
        this.updateAdvancedTimerUI();
      }
    }, 1000);
  }

  stopDisplayTicker() {
    if (this.displayTicker) {
      clearInterval(this.displayTicker);
      this.displayTicker = null;
    }
  }

  destroy() {
    this.stopPolling();
    this.stopDisplayTicker();
    
    if (this.islandElement) {
      this.islandElement.remove();
    }

    const sensor = document.getElementById('zsp-island-sensor');
    if (sensor) {
      sensor.remove();
    }
    // リスナーを削除
    chrome.runtime.onMessage.removeListener(this.broadcastListener);
    if (this._onStorageChanged) {
      try { chrome.storage.onChanged.removeListener(this._onStorageChanged); } catch (_) {}
      this._onStorageChanged = null;
    }
  }

  // 状態ブロードキャストをリッスン
  listenForBroadcasts() {
    this.broadcastListener = (message, sender, sendResponse) => {
      if (message.type === 'STATUS_BROADCAST') {
        const { kind, payload } = message;
        if (kind === 'timer') {
          this.timerState = payload || { active: false };
          // 実状態に合わせてモードを同期（compact表示の整合）
          if (this.timerState.active) {
            const mode = this.timerState.mode === 'manual' ? 'manual' : (this.timerState.mode || 'pomodoro');
            this.timerMode = mode;
            this.updateTimerModeButtons(mode);
            // 表示アンカーを更新
            this._displayAnchor.ts = Date.now();
            if (mode === 'manual') {
              this._displayAnchor.manualElapsedMs = this.timerState.currentElapsedMs || 0;
              this._displayAnchor.advRemainSec = 0;
            } else {
              this._displayAnchor.advRemainSec = this.timerState.remainSec || 0;
              this._displayAnchor.manualElapsedMs = 0;
            }
          }
          if (this.timerMode === 'manual') {
            this.updateFocusUI();
          } else {
            this.updateAdvancedTimerUI();
          }
        } else if (kind === 'lofi') {
          this.musicState.isPlaying = payload.isPlaying;
          this.musicState.currentTrackIndex = payload.trackIndex;
          this.musicState.volume = payload.volume;
          this.updateMusicDisplay();
          this.updateCompactStatus();
        } else if (kind === 'ambient') {
           this.updateAmbientSliders(payload.volumes);
        }
      }
    };
    chrome.runtime.onMessage.addListener(this.broadcastListener);
  }
  
  updateAmbientSliders(volumes) {
    const ambientSliders = this.islandElement.querySelectorAll('.zsp-slider[data-sound]');
    ambientSliders.forEach(slider => {
      const soundFile = slider.dataset.sound;
      const newVolume = volumes[soundFile] || 0;
      slider.value = newVolume;
      const volumeValue = slider.parentElement.querySelector('.zsp-volume-value');
      if (volumeValue) {
        volumeValue.textContent = `${newVolume}%`;
      }
    });
  }
}

// 初期化
let dynamicIsland = null;

function initDynamicIsland() {
  // N高等学校のサイトかチェック
  if (window.location.hostname === 'www.nnn.ed.nico') {
    // 設定を確認してから初期化
    chrome.storage.sync.get('showDynamicIsland', (result) => {
      const showDynamicIsland = result.showDynamicIsland !== false; // デフォルトで有効
      
      if (showDynamicIsland && !dynamicIsland) {
        dynamicIsland = new DynamicIsland();
      } else if (!showDynamicIsland && dynamicIsland) {
        // 無効にされた場合は破棄
        dynamicIsland.destroy();
        dynamicIsland = null;
      }
    });
  }
}

// グローバルにアクセス可能にする
window.initDynamicIsland = initDynamicIsland;

// content.jsの初期化を待つため、少し遅延させる
setTimeout(() => {
  // ページ読み込み完了後に初期化
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
      setTimeout(initDynamicIsland, 500); // content.jsの初期化後に実行
    });
  } else {
    setTimeout(initDynamicIsland, 500); // content.jsの初期化後に実行
  }
}, 100);

// ページ遷移時の再初期化
let dynamicIslandLastUrl = location.href;
new MutationObserver(() => {
  if (location.href !== dynamicIslandLastUrl) {
    dynamicIslandLastUrl = location.href;
    setTimeout(initDynamicIsland, 1000);
  }
}).observe(document, { subtree: true, childList: true });

// ページアンロード時のクリーンアップ
window.addEventListener('beforeunload', () => {
  if (dynamicIsland) {
    dynamicIsland.destroy();
    dynamicIsland = null;
  }
});

})(); // IIFE終了
